package PWx::MemberListPlot;

use warnings;
use strict;

use Params::Validate qw(:all);
use GD;
use GD::Image;
use PWx::TTT::Result;

=head1 NAME

PWx::MemberListPlot - Creates a graphical plot of a list of users.

=head1 VERSION

Version 0.1.0

=cut

use version;
our $VERSION = '0.1.0';

=head1 SYNOPSIS

This module creates a graphical plot of a list of users.

Perhaps a little code snippet.

    use PWx::MemberListPlot;

    my $foo = PWx::MemberListPlot->new();
    ...

=head1 FUNCTIONS

=head2 new

Creates a new plot object. This expects a few parameters as a hash.

	xaxis => one of EI, SN, TF, JP
	yaxis => one of EI, SN, TF, JP but a different one than x-axis
	users => arrayref to a list of PWx::User
	user  => the current user

Example:
	my $plot = new PWx::MemberListPlot(
		xaxis => 'EI',
		yaxis => 'SN',
		users => [@users],
		user  => $user);

=cut

sub new {
	my $class = shift;
	my %p     = validate(@_,
		{
			xaxis => { type => SCALAR, regex => qr/^(EI|SN|TF|JP)$/ },
			yaxis => { type => SCALAR, regex => qr/^(EI|SN|TF|JP)$/ },
			users => { type => ARRAYREF },
			user  => { isa  => 'PWx::Site::Platf::DB::Users' }
		}
	);
	
	my($width, $height) = (400, 300);
	my $img = GD::Image->new($width, $height);
	
	my $self = {
		width   => $width,
		height  => $height,
		padding => 30,
		scatter => 0.33,
		imgObj  => $img,
		mcolors => {
			0 => $img->colorAllocate(0, 0, 255),   # self
			1 => $img->colorAllocate(0, 0, 0),     # no_contact
			2 => $img->colorAllocate(0, 255, 0),   # in_contact
			3 => $img->colorAllocate(255, 0, 0),   # RCD_sent
			4 => $img->colorAllocate(255, 0, 255), # RCD_received
		},
		colors => {
			black      => $img->colorAllocate(0, 0, 0),
			background => $img->colorAllocate(255, 255, 255) 
		},
		%p
	};
	
	bless $self, $class;
	return $self;
}

=head2 data

Returns the image data, encodet as png.

	print $plot->data;

=cut

sub data {
	my $self = shift;
	$self->render;
	return $self->{imgObj}->png;
}

=head2 render

Renders the image. This is for internal use only.

=cut

sub render {
	my $self = shift;
	my $img  = $self->{imgObj};
	
	$img->filledRectangle(
		0,
		0,
		$self->{width},
		$self->{height},
		$self->{colors}->{background});
	
	my($areaX, $areaY) = $self->translate(0, 0);
	$img->rectangle(
		$areaX,
		$areaY,
		$self->{width}-$self->{padding},
		$self->{height}-$self->{padding},
		$self->{colors}->{black});
	
	#my($xmax, $ymax) = $self->calcMax;
	
	my $xstep = ($self->{height}-$self->{padding}*2)/4;
	foreach(0..4)
	{
		$img->line(
			$self->{padding}-5,
			$self->{padding}+$_*$xstep,
			$self->{padding},
			$self->{padding}+$_*$xstep,
			$self->{colors}->{black});
		$img->string(
			gdSmallFont,
			5,
			$self->{padding}+$_*$xstep-7,
			5*$_-10,
			$self->{colors}->{black});
	}
	
	my $ystep = ($self->{width}-$self->{padding}*2)/4;
	foreach(0..4)
	{
		$img->line(
			$self->{padding}+$_*$ystep,
			$self->{height}-$self->{padding}+5,
			$self->{padding}+$_*$ystep,
			$self->{height}-$self->{padding},
			$self->{colors}->{black});
		$img->string(
			gdSmallFont,
			$self->{padding}+$_*$ystep-2,
			$self->{height}-$self->{padding}+10,
			5*$_-10,
			$self->{colors}->{black});
	}
	
	my %rcds = map { $_->t_user_id => $_->rcd_type_id } $self->{user}->rcds;
	push @{$self->{users}}, $self->{user};
	
	foreach(@{$self->{users}})
	{
		next if !$_->result_id;
		my $tttRes = new PWx::TTT::Result($_->result->ttt);
		my $xwhich = "get$self->{xaxis}";
		my $ywhich = "get$self->{yaxis}";
		my $xoff   = $tttRes->$xwhich();
		my $yoff   = $tttRes->$ywhich();
		
		$xoff->[1] *= -1 if $xoff->[0] eq 'I';
		$xoff->[1] *= -1 if $xoff->[0] eq 'N';
		$xoff->[1] *= -1 if $xoff->[0] eq 'F';
		$xoff->[1] *= -1 if $xoff->[0] eq 'P';
		
		$yoff->[1] *= -1 if $yoff->[0] eq 'I';
		$yoff->[1] *= -1 if $yoff->[0] eq 'N';
		$yoff->[1] *= -1 if $yoff->[0] eq 'F';
		$yoff->[1] *= -1 if $yoff->[0] eq 'P';
		
		my $color = $self->{colors}->{black};
		
		$color = $self->{mcolors}->{0} if $_->id == $self->{user}->id;
		$color = $self->{mcolors}->{$rcds{$_->id}} if $rcds{$_->id};
		
		$img->filledEllipse(
			($self->{width}/2)  + $xoff->[1]*((($self->{width} - $self->{padding}*2)/2)/10)+$self->scatter,
			($self->{height}/2) + $yoff->[1]*((($self->{height} - $self->{padding}*2)/2)/10)+$self->scatter,
			10,
			10,
			$color);
	}
}

sub translate {
	my ($self, $x, $y) = @_;
	
	return ($x + $self->{padding}, $y + $self->{padding});
}

=head2 scatter

Returns an random scatter plus/minus 0.33 of the member icon size.
For internal use.

=cut

sub scatter {
	my $self = shift;
	return int(rand(10)*$self->{scatter});
}

=head2 calcMax

Calculates the maximum value for the x and y axes according
to the list of users. For internal use only.

=cut

sub calcMax {
	my $self = shift;
	
	my ($xmax, $ymax) = (0, 0);
	
	foreach(@{$self->{users}})
	{
		my $tttRes = new PWx::TTT::Result($_->result_id->ttt);
		my $xwhich  = "get$self->{xaxis}";
		my $ywhich  = "get$self->{yaxis}";
		
		if($tttRes->$xwhich->[1] > $xmax) {
			$xmax = $tttRes->$xwhich->[1]
		}
		if($tttRes->$ywhich->[1] > $ymax) {
			$ymax = $tttRes->$ywhich->[1]
		}
	}
	
	return ($xmax, $ymax);
}

=head1 AUTHOR

Johannes Klose, C<< <jk at plusw.de> >>

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc PWx::MemberListPlot

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright 2007 plusW Rolf Schaufelberger, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1; # End of PWx::MemberListPlot
