package PWx::Model;

use warnings;
use strict;

use version; our $VERSION = qv("0.1.0");
use Params::Validate qw (:all);
use PWx::Application;

# Testfunc for Params::Validate
sub _cdbiObject_or_index {
    my $v = shift;
    (ref($v) && UNIVERSAL::isa($v, 'PWx::DB') || !ref($v) && $v =~ /\d+/);
}

sub new {
    my $class = shift;

    my %p =
      validate_with(
          params => \@_,
          spec => { id => { callbacks => { 'id or class' => \&_cdbiObject_or_index } }, },
          allow_extra => 1,
          on_fail => sub { PWx::Exception->throw('Wrong or missing parameters', shift); },
      );

    my $self = {};

    # Retrieving db object
    if (ref($p{id})) {
        $self->{dbo} = $p{id};
    } else {
        my $s = PWx::Application->instance()->schema;

        (my $moniker = $class) =~ s/^.*:://;
        $self->{dbo} = $s->resultset($moniker)->find($p{id});
        return 0 unless $self->{dbo};
    }

    bless $self, $class;

    $self->_init(\%p);

    return $self;

}

sub search {
    my $class     = shift;
    my $param     = shift;
    my $initparam = shift;

    my $s = PWx::Application->instance()->schema;
    (my $moniker = $class) =~ s/^.*:://;
    my $rs = $s->resultset($moniker)->search($param);

    if ($rs) {
        my $self = { dbo => $rs->first };
        bless $self, $class;
        $self->_init($initparam);
        return $self;
    } else {
        PWx::Exception->throw("No record found");
    }
}

sub create {
    my $class     = shift;
    my $param     = shift;
    my $initparam = shift;

    foreach (keys %$param) {
        $param->{$_} = undef if ($param->{$_} eq "");
    }

    my $s = PWx::Application->instance()->schema;
    (my $moniker = $class) =~ s/^.*:://;
    my $rs = $s->resultset($moniker)->create($param);

    if ($rs) {
        my $self = { dbo => $rs };
        bless $self, $class;
        $self->_init($initparam);
        return $self;
    } else {
        PWx::Exception->throw("Error creating record: $@");
    }
}

sub _init { }

sub getDbObject {
    return shift->{dbo};
}

sub AUTOLOAD {
    my $self = shift;
    our $AUTOLOAD;
    my $method = $AUTOLOAD;
    $method =~ s/.*:://;
    return if $method eq 'DESTROY';
    $self->{dbo}->$method(@_);
}

1;    # End of code PWx::Model

__END__

=head1 NAME

PWx::Model -     Base class for Model classes 

=head1 VERSION

$Id$

=head1 SYNOPSIS

    package Pwx::MyModel;

    use base qw/PWx::Model/;

    my $foo = PWx::MyModel->new(id => id);
    ...


=head1 METHODS

=head2 new 
 
  new (id => $id, @extra )

  id = id of Object in Database or already retrieved Db object
  Database object is stored in {dbo} 
  Calls _init (@extra) 

=head2 search ( %params ) 

  Not Yet implemented

=head2 _init 

 _init ( %params ) 

  To be overwritten in model class 

=head2 getDbObject

  Returns the database Object

=head2 AUTOLOAD

  Maps all DBIx::Class column methods to the dbo object

  
=head1 AUTHOR

Rolf Schaufelberger, C<< <rs@plusw.de> >>

=head1 COPYRIGHT & LICENSE

Copyright 2006-2007 plusW, Rolf Schaufelberger.  All Rights Reserved.

As this program is no free software, you are not allowed to redistribute
and/or sell it, not even to modify or copy it.

=cut
