package PWx::TTT;

use warnings;
use strict;

use version; our $VERSION = qv("0.2.0");
use Params::Validate qw(:all);
use PWx::Application;
use Log::Log4perl qw (get_logger);
use PWx::Exceptions;

=head1 NAME

PWx::TTT - Documentation for  PWx::TTT

  PWx::TTT handles the logic for TTT answers  comutes and stores the result

=head1 VERSION

$Id$

=head1 SYNOPSIS

Short description of the module.

    use PWx::TTT;

    my $ttt = PWx::TTT->new(answers => $anwers);
    $ttt->checkRequirements ();
    $ttt->saveResult();


=head2 new ( answers => $answers, user_id => $user_id )

=head3 Parameters

 $answers = Arrayref, holding the answers as stored in the session

=cut


sub new {
  my $class = shift;
  my %param = validate_with (params => \@_,
			     spec  => { answers => {type => HASHREF},
				       user_id => {type => SCALAR},
				     } ) ;

  my $self = { %param };
  my $app = PWx::Application->instance();
  $self->{schema} = $app->schema;

  bless $self, $class;
  $self->_computeResult();

  return $self;
}

=pod

=head2 checkRequirements ()

=head3 Returns

 0 on failure 1 on succes


=cut

sub checkRequirements {
  my $self = shift;

  foreach ( keys %{$self->{dims}} ) {
    if ( $self->{dims}->{$_}{cnt} < 5 ) {
      return 0 ;
    }
  }
  return 1;
}

=pod

=head2 savsResult()

=head3 Returns

 void

=head3 Exceptions

 none

=cut

sub saveResult {
  my $self = shift;

  my $mbti = $self->{result};
  $mbti =~ s/[^A-Z]//g;

  my $result = $self->{schema}->resultset('Results')->create({ 
				  user_id => $self->{user_id},
				  ttt     => $self->{result},
				  mbti    => $mbti } );
  $self->{schema}->txn_commit;
}

=pod

=head2 _computeResult()

=head3 Returns

 string

=head3 Exceptions

 none

=cut

sub _computeResult {
  my $self = shift;
  my $result = '';

#  my $logger = get_logger('PWx::TTT');

  my $q = $self->_readQuestions();
  my $dims = $self->_readDimensions();

  foreach ( keys %{$self->{answers}} ) {
    my $idx = substr($_, 2);   # A.12 
    my $answer = $self->{answers}->{$_};
    my $dimgrp = $q->{$idx}->dimension;
    my $answ_idx =  $answer == 1 ? $q->{$idx}->answer1 : $q->{$idx}->answer2;
    $dims->{$dimgrp}{$answ_idx}{cnt}++;     # individual sum
    $dims->{$dimgrp}{cnt}++;                # dimension sum
  }

  foreach (sort keys %$dims ) {
    my @dimensions =  keys %{$dims->{$_}} ;
    my $diff = $dims->{$_}->{$dimensions[0]}{cnt} - $dims->{$_}->{$dimensions[1]}{cnt} ;
    if ($diff > 0  ) { 
      $result .= $dimensions[0] . '+' . $diff;
    } elsif ($diff < 0) {
      $result .= $dimensions[1] . '+' . (-1) * $diff;
    } elsif ($dims->{$_}->{$dimensions[0]}{sort} - $dims->{$_}->{$dimensions[1]}{sort} > 0 ) {
      $result .=  $dimensions[1] . '+0';
    } else {
      $result .=  $dimensions[0] . '+0';
    }
  }
  $self->{dims}   = $dims;
  $self->{result} = $result;
}

=pod

=head2 _readQuestions()

=head3 Returns

 hashref with Question Objects

=head3 Exceptions

 none

=cut

sub _readQuestions() {
  my $self = shift;
  my %q;

  my $rs = $self->{schema}->resultset('Questions')->search();
  while ( my $d = $rs->next ) {
    $q{$d->id} =  $d;
  }
  return \%q;
}

=pod

=head2 _readDimensions()

=head3 Returns

 hashref with Dimensions

=head3 Exceptions

 none

=cut

sub _readDimensions() {
  my $self = shift;
  my %dims;

  my $rs = $self->{schema}->resultset('Dimensions')->search();
  while ( my $d = $rs->next ) {
    $dims{$d->dimension}{$d->answer} = { sort => $d->sort,
					  cnt  => 0
					};
  }
  return \%dims;
}

1; # End of code PWx::TTT

__END__

=head1 AUTHOR

Rolf Schaufelberger, C<< <rs@plusw.de> >>

=head1 COPYRIGHT & LICENSE

  This program is free software; you can redistribute it and/or modify it
  under the same terms as Perl itself.

