package PWx::WebApp;

use warnings;
use strict;

use Data::FormValidator;
use Data::FormValidator::Constraints qw(:closures);
use Digest::SHA1;
use constant MAC_SECRET => 'MySecret';
use PWx::Users;
use PWx::Users::Guest;
use HTML::Entities;

sub _set_user_cookie {
    my $self   = shift;
    my $userid = shift;

    my $value = {
                  user_id => $userid,
                  MAC     => Digest::SHA1::sha1_base64($userid, MAC_SECRET),
                };

    $self->_set_cookie($self->param->UserCookieName, $value, '+1h');
}

sub _set_user {
    my $self = shift;

    $self->{current_user} = $self->user_from_cookie();

    if ($self->{current_user}->is_guest) {
        $self->_unset_user_cookie;
    }

#    $self->{DefaultCustomerId} = $self->{current_user}->company_id->id;
}

sub _unset_user_cookie {
    my $self = shift;
    $self->_set_cookie($self->param->UserCookieName, '', '-1d');
}

sub user_from_cookie {
    my $self = shift;

    if (Apache::Cookie->fetch
        && (my $cookie = Apache::Cookie->fetch->{ $self->param->UserCookieName }))
    {
        my %data = $cookie->value;

        if ($data{user_id}) {
            my $mac = Digest::SHA1::sha1_base64($data{user_id}, MAC_SECRET);
            if ($data{MAC} eq $mac) {
                my $u = PWx::Users->new(id => $data{user_id});

                $self->logger->debug("Create user from Cookie", $data{user_id});
                $self->logger->debug("User = ",                 $u ? $u->id : 'failed');

                return $u ? $u : PWx::Users::Guest->new();
            }
        }
    }
    $self->logger->debug("Create guest user, no cookie found");
    return PWx::Users::Guest->new();
}

sub User_Login {
    my $self     = shift;
    my $username = lc($self->args->{username});

    if ($self->args->{register}) {
        $self->redirect(path => '/register.html');
    } elsif ($self->args->{forgotpw}) {
        $self->User_ForgotPW();
    } else {
        my $user = $self->schema->resultset('Users')->search(
                                                      username => $username,
                                                      password => $self->args->{password},
        )->single;

        unless ($user) {
            $self->_handle_error(
                                 error     => 'Invalid email address or password',
                                 uri       => '/',
                                 save_args => {
                                                username    => $username,
                                                redirect_to => $self->args->{redirect_to},
                                              },
                                );
        }

        $self->logger->debug("username => $username, password => ",
                             $self->args->{password});
        $self->logger->debug("User = ", $user->id);
        $self->delete_session;
        $self->_set_user_cookie($user->id);
        $self->_redirect_from_args('/');
    }
}

sub User_Logout {
    my $self = shift;

    if ($self->current_user->is_logged_in) {
        $self->current_user->last_logout("now()");
        $self->current_user->update;
    }

    $self->_unset_user_cookie;
    $self->_unset_user_cookie;
    $self->delete_session;
    $self->_add_message('Logout.LoggedOut');
    $self->_redirect_from_args('/');
}

#
# To send a forgotten password is not needed by specifications
#
#sub User_ForgotPW {
#    my $self = shift;
#    my $dbuser;
#
#    if ($self->args->{email}) {
#        $dbuser =
#          $self->schema->resultset('Users')->search(username => $self->args->{username});
#    } elsif ($self->args->{firstname} && $self->args->{lastname}) {
#        $dbuser = $self->schema->resultset('Users')->search(
#                                                    firstname => $self->args->{firstname},
#                                                    lastname  => $self->args->{lastname});
#    }
#
#    unless ($dbuser && $dbuser->count) {
#        $self->_add_message('Password.Forgotten.NoSuchUser');
#        $self->_handle_error(error => 'No user matches this email address.',
#                             uri   => '/forgot_password.html',);
#    }
#
#    my $user = PWx::Users->new(user => $dbuser->first);
#    $self->apache_req->register_cleanup(sub { $user->forgot_password });
#    $self->_add_message('Password.Forgotten.Sent');
#    $self->redirect(path  => '/forgot_password.html',
#                    query => { show_form => 0 });
#}

sub user_id_from_cookie {
    my $user = user_from_cookie;
    return $user->id || 0;
}

sub User_Register {
    my $self = shift;

    my $profile = {
           required =>
             [qw(username password firstname lastname email city country_id life_motto)],
           optional => [qw(life_motto2 likes dislikes gps enneagram1 enneagram2)],
           constraint_methods => {
               email    => email(),
               gps      => qr/^\d+(\.\d+)? ?[NnSs] ?,? ?\d+(\.\d+)? ?[EeWw]$/,
               username => _check_username_existence($self, { fields => [qw(username)] }),
           },
    };

    my $result = Data::FormValidator->check($self->args, $profile);

    if ($result->has_missing || $result->has_invalid) {
        my %missing = map { $_ => 1 } @{ $result->missing } if $result->has_missing;
        $self->_handle_error(
                             uri       => '/register.html',
                             error     => 'Register.Failed',
                             save_args => {
                                            missing => \%missing,
                                            invalid => scalar $result->invalid,
                                            args    => $self->args
                                          }
                            );
    } else {
        my $dbargs = $self->args;
        delete $dbargs->{register};
        $dbargs->{register_date} = "now()";

        $dbargs->{$_} = encode_entities($dbargs->{$_}) foreach (keys %$dbargs);

        my $newuser = PWx::Users->create($dbargs);
        $self->_set_user_cookie($newuser->id);
        $self->redirect(uri => "/");
    }
}

sub _check_username_existence {
    my ($self, $attrs) = @_;
    my $username = $attrs->{fields}->[0] if $attrs->{fields};

    return sub {
        my $dfv = shift;

        # Name it to refer to in the 'msgs' system.
        $dfv->name_this('check_username_existence');

        my $val = $dfv->get_current_constraint_value();

        my $user = $self->schema->resultset('Users')->search(username => $val)->single;

        return $user ? 0 : 1;
      }
}

sub User_Delete {

}

sub User_Edit {

}

sub User_Search {
    my $self = shift;

}

1;

__END__

=head1 NAME

PW::App::User - Dokumentation zu PW::WebApp::User!

=head1 VERSION

$Id$

=head1 METHODS

=head2 _set_user_cookie 

=head2 _set_user 

 Setzt die Variablen current_user und DefaultCustomerId.
 DefaultCustomerId wird aus dem Paramater  *::Param::DefaultUser bestimmt.

=head2 _unset_user_cookie 

=head2 _user_from_cookie 

=head2 _user_id_from_cookie

=head2 User_Create 

=head2 User_Delete 

=head2 User_Edit 

=head2 User_Search 


=head1 AUTHOR

Rolf Schaufelberger, C<< <rs@plusw.de> >>

=head1 COPYRIGHT & LICENSE

Copyright 2006 plusW, Rolf Schaufelberger.  All Rights Reserved.

As this program is no free software, you are not allowed to redistribute
and/or sell it, not even to modify or copy it.
