package PbT;

use strict;

our $VERSION = '0.01';

use base 'GENPbT';

use Cache::FastMmap;
use BerkeleyDB;

use PbT::UserData;
use PbT::LikeData;
use PbT::TTQuestion;
use PbT::TTAnswer;
use PbT::TTSummary;
use PbT::Enne;
use PbT::Visible;
use PbT::Model;

use Gantry::Utils::CRUDHelp;
use Gantry::Plugins::AuthCookie;
use Gantry::Plugins::DBIxClassConn qw( get_schema );

#-----------------------------------------------------------------
# schema_base_class
#-----------------------------------------------------------------
sub schema_base_class { return 'PbT::Model'; }

#-----------------------------------------------------------------
# $self->namespace(  )
#-----------------------------------------------------------------
sub namespace {
	return 'pbt';
}

#-----------------------------------------------------------------
# $self->init( $r )
#-----------------------------------------------------------------
sub init {
    my ( $self, $r ) = @_;

    # process SUPER's init code
    $self->SUPER::init( $r );
    
    # setup berkeley DB for sessions
    my $db_filename = "/tmp/pbt.db";
    my  %db_sessions;
    tie %db_sessions, "BerkeleyDB::Hash",
        -Filename => $db_filename,
        -Flags    => DB_CREATE
    or die "Cannot open file $db_filename: $! $BerkeleyDB::Error\n" ;
                
    $self->set_db_session( \%db_sessions );
        
    
} # END init

#-----------------------------------------------------------------
# $self->gps_split_numberic( coords )
#-----------------------------------------------------------------
sub gps_split_numeric {
    my( $self, $coords ) = @_;
    
    my( $lng, $lat ) = ( '', '' );

    # split and collect lat lng        
     ( $lng, $lat ) = ( $coords =~ /^(.*?[ns])\s*,?\s*(.*?[ew])/i );

     if ( $lng =~ s/\s*N\s*//i ) {
         $lng = "-$lng";
     }
     else {
         $lng =~ s/\s*[ns]\s*$//i;
     }
     
     if ( $lat =~ s/\s*W\s*//i ) {
         $lat = "-$lat";
     }
     else {
         $lat =~ s/\s*[ew]\s*$//i;
     }

    return( $lng, $lat );
}

#-----------------------------------------------------------------
# $self->gps_display( lng, lat)
#-----------------------------------------------------------------
sub gps_display {
    my( $self, $lng, $lat ) = @_;
    
    return if ( ! $lng || ! $lat );
        
    if ( $lng =~ s/^\-//i ) {
         $lng = "$lng N";
     }
     else {
         $lng = "$lng S";
     }
     
     if ( $lat =~ s/^\-//i ) {
         $lat = "$lat W";
     }
     else {
         $lat = "$lat E";
     }

    return( "$lng, $lat" );
}


#-----------------------------------------------------------------
# $self->in_contact
#-----------------------------------------------------------------
sub in_contact {
    my ( $self, $user_id, $contact_id ) = @_;
      
    my $sch   = $self->get_schema();     
    my $contact = $sch->resultset( 'contacts' )->search( {
        user_id => $user_id, contact_id => $contact_id 
    } )->next;
    
    if ( $contact ) {
        return 1;
    }
    
    return 0;
    
}

#-----------------------------------------------------------------
# $self->do_main(  )
#-----------------------------------------------------------------
sub do_main {
    my ( $self ) = @_;

    $self->stash->view->template( 'main.tt' );
    $self->stash->view->title( 'PbT' );
    
    my %param = $self->get_param_hash();  
    my $sch   = $self->get_schema();     
    
    my $errors = {};
    if ( $self->is_post() && ! ( $errors = $self->register_checkvals() ) ) {                
            
        # remove non-essential fields
        clean_params( \%param );
        
        # split and collect lat lng        
         ( $param{'GPS_longitude'}, $param{'GPS_latitude'} ) 
             = $self->gps_split_numeric( $param{gps_coords} );
        
        my @likes    = split( /\s*,\s*/, $param{list_of_likes} );
        my @dislikes = split( /\s*,\s*/, $param{list_of_dislikes} );

        foreach ( qw/page username submit gps_coords list_of_dislikes list_of_likes/ ) {
            delete( $param{$_} );
        }
                            
        my $i = $sch->resultset( 'user_data' )->create( \%param );
        
        my $user_id = $i->id;

        if ( scalar( @likes ) > 0 ) {
            $self->make_likes( $user_id, \@likes, 't' );
        }

        if ( scalar( @dislikes ) > 0 ) {
            $self->make_likes( $user_id, \@dislikes, 'f' );
        }
        
		my $encd = Gantry::Plugins::AuthCookie::encrypt_cookie( 
		    $self, 
		    $param{user_name}, 
		    $param{password} 
		);

		# set cookie, redirect to do_frontpage.
        $self->set_cookie( {  
            name     => 'auth_cookie', 
            value    => $encd, 
            path     => '/',
        } );
        
        $self->relocate( $self->app_rootp . "/home" );
        return;
    }
        
    my @countries = $sch->resultset( 'country' )->search( 
        undef
    );
    
    my @enne_list = $sch->resultset( 'enne' )->search( 
        undef,
        { order_by => 'description' }
    );   
     
    $self->stash->view->data( {
        enne_list   => \@enne_list,
        errors      => $errors,
        countries   => \@countries,
        pages       => $self->site_links()
    } );
    
} # END do_main

sub get_likes {
    my ( $self, $user_id, $like_or_dislike ) = @_;

    my $sch   = $self->get_schema();     

    my @list;
    my $likes = $sch->resultset( 'user_data_like_data' )->search(
        { user_data => $user_id, like_or_dislike => $like_or_dislike }
    );

    while ( my $like_row = $likes->next ) {
        push( @list, $like_row->like_data->like_text );        
    }    

    return join( ', ', @list );

}

sub make_likes {
    my ( $self, $user_id, $likes, $like_or_dislike ) = @_;

    my $sch   = $self->get_schema();     

    if ( scalar( @{ $likes } ) > 0 ) {
        my $del = $sch->resultset( 'user_data_like_data' )->search(
            { user_data => $user_id, like_or_dislike => $like_or_dislike }
        )->delete;
    
        foreach my $like_text ( @{ $likes } ) {
    
            my $like = $sch->resultset( 'like_data' )->find_or_create(
                like_text => lc( $like_text )
            );
        
            my $like_rel = $sch->resultset( 'user_data_like_data' )->create(
                {
                    like_data       => $like->id,
                    user_data       => $user_id,
                    like_or_dislike => $like_or_dislike,
                }
            );
        }
    }
}

#-------------------------------------------------
# $self->mycache
#-------------------------------------------------
sub mycache {
    my( $self ) = @_;
	
	my $cache_file = __PACKAGE__;
	$cache_file =~ s/\:\:/_/g;

    return(
         Cache::FastMmap->new(
            share_file => "/tmp/$cache_file",
            expire_time => 3600
         )
    );
}

#-----------------------------------------------------------------
# $self->set_db_session( $new_value )
#-----------------------------------------------------------------
sub set_db_session {
    my ( $self, $value ) = @_;

    $self->{ __DB_SESSIONS__ } = $value;
}

#-----------------------------------------------------------------
# $self->db_session( $new_value )
#-----------------------------------------------------------------
sub db_sessions {
    my ( $self, $value ) = @_;

    return( $self->{ __DB_SESSIONS__ } );
}

#-----------------------------------------------------------------
# $self->site_links(  )
#-----------------------------------------------------------------
sub site_links {
    my ( $self ) = @_;

    return [
        { link => $self->app_rootp() . '/search', label => 'Search' },
        { link => $self->app_rootp() . '/status', label => 'Status' },
        { link => $self->app_rootp() . '/home', label => 'Home' },
        { link => $self->app_rootp() . '/admin/user', label => 'User' },
        { link => $self->app_rootp() . '/admin/like', label => 'Like' },
        { link => $self->app_rootp() . '/admin/tt_question', label => 'Tt Question' },
        { link => $self->app_rootp() . '/admin/tt_answer', label => 'Tt Answer' },
        { link => $self->app_rootp() . '/admin/tt_summary', label => 'Tt Summary' },
        { link => $self->app_rootp() . '/admin/enne', label => 'Enne' },
        { link => $self->app_rootp() . '/admin/visible', label => 'Visible' },
    ];
} # END site_links

#-----------------------------------------------------------------
# $self->register_checkvals(  )
#-----------------------------------------------------------------
sub register_checkvals {
    my( $self ) = @_;

    my %param  = $self->get_param_hash();
    my $errors = {};  
     
    foreach ( qw/full_name user_name password email town country life_motto/ ) {
        $errors->{$_} = 'Required' if ! $param{$_};
    }

    if ( $param{user_name} ) {

        if ( length( $param{user_name} ) < 4  || length( $param{user_name} ) > 16 ) {
            $errors->{user_name} = 'username must be between 4 and 16 characters'
        }      
        else {
            my $sch = $self->get_schema();    
            my @user = $sch->resultset( 'user_data' )->search( 
                user_name => $param{user_name},
            );
            if ( $user[0] ) {
                $errors->{user_name} = "Username already taken, choose another";
            }   
        }
    }

    if ( $param{password} ) {
        if ( length( $param{password} ) < 2  
            || length( $param{password} ) > 16 )  {
            
            $errors->{password} = 'username must be between 2 and 16 characters'
        }
    }       

    if ( $param{gps_coords} ) {
        if ( $param{gps_coords} !~ /^\d+\.?\d*\ ?[ns]\ ?,?\ ?\d+\.?\d*\ ?[ew]/i ) {
            $errors->{gps_coords} = 'invalid format for GPS Coordinates ';            
        }
    }
    
    return( $errors ) if %{ $errors } > 0;
}

1;

=head1 NAME

PbT - the base module of this web app

=head1 SYNOPSIS

This package is meant to be used in a stand alone server/CGI script or the
Perl block of an httpd.conf file.

Stand Alone Server or CGI script:

    use PbT;

    my $cgi = Gantry::Engine::CGI->new( {
        config => {
            #...
        },
        locations => {
            '/' => 'PbT',
            #...
        },
    } );

httpd.conf:

    <Perl>
        # ...
        use PbT;
    </Perl>

If all went well, one of these was correctly written during app generation.

=head1 DESCRIPTION

This is the base controller of the app.  It presents the registration/login
page.

=head1 METHODS

=over 4

=item schema_base_class

=item get_orm_helper

=back


=head1 METHODS INHERITED FROM GENPbT or its plugins

=over 4

=item init

=item do_main

=item db_sessions

Returns a hash reference tied to the session BerkleyDB database.

=item set_db_session

For use during init, to set up session db.

=item site_links

=item mycache

=item namespace

=item register_checkvals

=item gps_display

=item gps_split_numeric

=item make_likes

Creates like and dislike rows and the joining rows between them and the
user_data table.

=item schema_base_class

=back


=head1 SEE ALSO

    Gantry
    GENPbT
    PbT::User
    PbT::Like
    PbT::TTQuestion
    PbT::TTAnswer
    PbT::TTSummary
    PbT::Enne
    PbT::Visible

=head1 AUTHOR

Phil Crow, E<lt>phil@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2007 Phil Crow

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut
