package PbT::Home;

use strict;
use Gantry::Utils::CRUDHelp;

use base 'PbT';

use PbT::Model;
sub schema_base_class { return 'PbT::Model'; }
use Gantry::Plugins::DBIxClassConn qw( get_schema );

##-----------------------------------------------------------------
## $self->init( $r )
##-----------------------------------------------------------------
sub init {
    my ( $self, $r ) = @_;

    # process SUPER's init code
    $self->SUPER::init( $r );

    $self->auth_deny( 'yes' );
    
} # END init

#-----------------------------------------------------------------
# $self->do_default( )
#-----------------------------------------------------------------
sub do_main {
    my ( $self, $user_id ) = @_;
    
    $self->stash->view->template( 'home.tt' );    
    
    $user_id ||= $self->auth_user_row->id;
    
    my %param = $self->get_param_hash();        
    my $sch   = $self->get_schema();
    $param{page} ||= 1;

    my $user_row = $sch->resultset( 'user_data' )->search( {
        id => $user_id,
    } )->next;
    
    die "invalid user" if ! $user_row;
    
    my $summary_list = $sch->resultset( 'tt_summary' )->search( 
        { 
            user_data => $user_id,
             
        },
        {
            order_by => 'active DESC, created DESC',
   			page => $param{page},
    		rows => $self->fish_config( 'tests_per_page' ),
        }
    );

    my $contact_list = $sch->resultset( 'contacts' )->search( 
        { 
            user_id => $user_id,  
        },
    );

    my $received_messages = $sch->resultset( 'visible' )->search(
        {
            user_rec => $user_id,
            status   => 1,
        },
    );

    my $gps_coords = $self->gps_display( 
        $user_row->GPS_longitude, $user_row->GPS_latitude 
    );

    $self->stash->view->data( {
        gps_coords   => $gps_coords,
        list_of_likes        => $self->get_likes( $user_id, 't' ),
        list_of_dislikes     => $self->get_likes( $user_id, 'f' ),
        user_row     => $user_row,
        contact_list => $contact_list, 
        summary_list => $summary_list,
        page         => $summary_list->pager,

        received_messages => $received_messages,
     } );      
}

sub do_accept_rec {
    my( $self, $rec_id ) = @_;

    my %param = $self->get_param_hash();        
    my $sch   = $self->get_schema();
    
    my $r = $sch->resultset( 'visible' )->find( $rec_id );
    
    die "invalid rec" if ! $r;
    
    $r->update( { status => 2 } );
    
    $self->relocate( $self->app_rootp . "/home");
}

sub do_reject_rec {
    my( $self, $rec_id ) = @_;

    my %param = $self->get_param_hash();        
    my $sch   = $self->get_schema();
    
    my $r = $sch->resultset( 'visible' )->find( $rec_id );
    
    die "invalid rec" if ! $r;
    
    $r->update( { status => 3 } );
    
    $self->relocate( $self->app_rootp . "/home" );
    
}

#-----------------------------------------------------------------
# $self->do_edit( )
#-----------------------------------------------------------------
sub do_edit {
    my ( $self ) = @_;

    my $user_id = $self->auth_user_row->id;
    my %param   = $self->get_param_hash();

    $self->stash->view->template( 'home_edit.tt' );

    my $sch = $self->get_schema();
    my $row = $sch->resultset( 'user_data' )->search( 
        { id => $user_id } 
    )->next;
    
    my $errors = {};
    if ( $self->is_post() && ! ( $errors = $self->checkvals() ) ) {                

        # update dislikes
        my @dislikes = split( /\s*,\s*/, $param{list_of_dislikes} );
        my @likes    = split( /\s*,\s*/, $param{list_of_likes} );

        $self->make_likes( $user_id, \@dislikes, 'f' );
        $self->make_likes( $user_id, \@likes, 't' );
         
        ( $param{'GPS_longitude'}, $param{'GPS_latitude'} ) 
            = $self->gps_split_numeric( $param{gps_coords} );    
        
        # remove non-essential fields
        clean_params( \%param );
        delete( $param{submit} );
        delete( $param{gps_coords} );
        delete( $param{list_of_likes} );
        delete( $param{list_of_dislikes} );
        
        # add modified date
        $param{modified} = 'now';

        $row->update( \%param );
    
        $self->relocate( $self->app_rootp . "/home" );
        return;
    }
    
    my @countries = $sch->resultset( 'country' )->search( 
        undef
    );
    
    my $gps_coords = $self->gps_display( $row->GPS_longitude, $row->GPS_latitude );
 
    my @enne_list = $sch->resultset( 'enne' )->search( 
        undef,
        { order_by => 'description' }
    );   
    
    $self->stash->view->data( {
        enne_list   => \@enne_list,
        errors      => $errors,
        countries   => \@countries,
        row         => $row,
        list_of_likes       => $self->get_likes( $user_id, 't' ),,
        list_of_dislikes    => $self->get_likes( $user_id, 'f' ),,
        gps_coords          => $gps_coords,        
    } );

}

sub checkvals {
    my( $self ) = @_;
    
    my %param  = $self->get_param_hash();
    my $errors = {};  
     
    foreach ( qw/full_name email town country life_motto/ ) {
        $errors->{$_} = 'Required' if ! $param{$_};
    }

    if ( $param{gps_coord} ) {
        if ( $param{gps_coord} !~ /^\d+\.?\d*\ ?[ns]\ ?,?\ ?\d+\.?\d*\ ?[ew]/i ) {
            $errors->{gps_coord} = 'invalid format for GPS Coordinates ' . $param{gps_coord};            
        }
    }
    
    return( $errors ) if %{ $errors } > 0;    
}

1;

=head1 NAME

PbT::Home - A search controller in the PbT application

=head1 SYNOPSIS

This package is meant to be used in a stand alone server/CGI script or the
Perl block of an httpd.conf file.

Stand Alone Server or CGI script:

    use PbT::Home;

    my $cgi = Gantry::Engine::CGI->new( {
        config => {
            #...
        },
        locations => {
            '/search' => 'PbT::Home',
            #...
        },
    } );

httpd.conf:

    <Perl>
        # ...
        use PbT::Home;
    </Perl>

    <Location /search>
        SetHandler  perl-script
        PerlHandler PbT::Home
    </Location>

If all went well, one of these was correctly written during app generation.

=head1 DESCRIPTION

This module presents the user's home page (also called status page) and
allows users to change personal data.

=head1 METHODS

=over 4

=item init

=item do_main

Presents the home page (also called the status page).

=item do_edit

Allows users to update their personal info.

=item schema_base_class

=item checkvals

For internal use during form validation.

=back


=head1 DEPENDENCIES

    PbT
    Gantry::Utils::CRUDHelp

=head1 AUTHOR

Phil Crow, E<lt>phil@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2007 Phil Crow

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut
