package PbT::Search;

use strict;
use Gantry::Utils::CRUDHelp;

use base 'PbT';

use PbT::Model;
sub schema_base_class { return 'PbT::Model'; }
use Gantry::Plugins::DBIxClassConn qw( get_schema );

# CPAN modules 
use Chart::Graph::Gnuplot qw( gnuplot ); 
use File::Temp; 
use File::Slurp; 

# Regexp that will validate our graph types 
my $valid_axis_type_re = qr/^
                            (
                                EI          |
                                SN          |
                                TF          |
                                JP          |
                                SJ          |
                                SP          |
                                NT          |
                                NF          |
                                distance    |
                                likes       |
                                age         |
                                TTTage      |
                            )
                         $/ox; 

my $graph_ranges = { 
    'EI'        =>  '[-10:10]', 
    'SN'        =>  '[-10:10]', 
    'TF'        =>  '[-10:10]', 
    'JP'        =>  '[-10:10]', 
}; 

my $axis_selects = { 
    'EI'        =>  'extrovert', 
    'SN'        =>  'sensing',
    'TF'        =>  'thinking', 
    'JP'        =>  'judging', 
}; 

#-----------------------------------------------------------------
# $self->do_main( )
#-----------------------------------------------------------------
# On GET requests simply display the search form, otherwise
# process the search results 
#-----------------------------------------------------------------
sub do_main {
    my $self    =   shift; 
    
    my $ins     = $self->params();        
    my $s       = $self->get_schema();

    if ( $self->is_post() ) {                
     
        # Normal search arguments 
        my $args = { 
                    type                =>  'HTML',
                    limit               =>  50,
                    offset              =>  0, 
                    no_contact          =>  $ins->{no_contact},
                    no_request          =>  $ins->{no_request},
        }; 

        # Handle life motto searches 
        if( $ins->{life_motto_contains} ) { 
            $args->{life_motto_contains} = $ins->{life_motto_contains_search}; 
        }

        # Handle searches for TTT types 
        if( $ins->{ttt_types} ) { 
            $args->{ttt_types} = join( '', 
                                            $ins->{ei_types},
                                            $ins->{sn_types},
                                            $ins->{tf_types},
                                            $ins->{jp_types} ); 
        } 

        my $data_ref = $self->perform_search( $args ); 

        $self->stash->view->template( 'search_results.tt' );              
        $self->stash->view->data( {
                                    results => $data_ref,
                                    users   => join(',', 
                                                    keys( %{ $data_ref } ) ),
                                  } ); 

        return();
    }
    
    $self->stash->view->template( 'search.tt' );    
    $self->stash->view->data( { } );        
}

#-----------------------------------------------------------------
# $self->determine_relationship
#-----------------------------------------------------------------
# This should probably be moved up to the app level module later. 
# This determines what the relationship to the current user 
#-----------------------------------------------------------------
sub _determine_relationship { 
    my $self        =   shift; 
    my $id          =   shift; 

    warn( "Relationship ID: $id "); 

    # Get our current user's ID 
    my $current_user    = $self->auth_user_row->id; 
    my $dbh             = $self->get_schema()->storage->dbh; 

    # Check incontact
    my $contact_sth = $dbh->prepare(
        qq{
            SELECT contact_id FROM contacts WHERE user_id = $current_user
            AND contact_id = $id
        }
    ) or die "Bad incontact: $!";
    $contact_sth->execute  or die "Incontact execute: $!";

    if( $contact_sth->fetchrow ) { 
        $contact_sth->finish; 
        return( 'incontact' ); 
    }

    $contact_sth->finish; 

    # Check sent status 
    my $sent_sth = $dbh->prepare(
        qq{
            SELECT id FROM visible
            WHERE user_send = $current_user AND 
                  user_rec  = $id
        }
    ) or die "Bad Sent: $!";
    $sent_sth->execute or die "Sent execute: $!"; 

    if( $sent_sth->fetchrow ) { 
        $sent_sth->finish; 
        return( 'sent' ); 
    }
    $sent_sth->finish; 

    # Check rec status 
    my $rec_sth = $dbh->prepare(
        qq{
            SELECT id FROM visible
            WHERE user_rec = $current_user AND 
                  user_send  = $id 
        }
    ) or die "Bad Rec: $!";
    $rec_sth->execute or die "Rec execute: $!"; 

    if( $rec_sth->fetchrow ) { 
        $rec_sth->finish; 
        return( 'received' ); 
    }
    $rec_sth->finish; 

    # Default to no contact
    return( 'nocontact' ); 

} # END _determine_relationship

#-----------------------------------------------------------------
# $self->_scatter
#-----------------------------------------------------------------
# This function scatters our integer data by +/- .33 randomly
#-----------------------------------------------------------------
sub _scatter { 
    my $self    =   shift; 
    my $x_ref   =   shift; 
    my $y_ref   =   shift; 

    my $do_x = int(rand(50)); 
    my $do_y = int(rand(50)); 

    my $x_plus_minus = int(rand(50)); 
    my $y_plus_minus = int(rand(50)); 

    if( $do_x < 23 ) { 

        $$x_ref += .33 if ( $x_plus_minus <= 18 ); 
        $$x_ref -= .33 if ( $x_plus_minus >= 33 ); 
    }
    else {

        $$x_ref += .33 if ( $x_plus_minus <= 27 ); 
        $$x_ref -= .33 if ( $x_plus_minus >= 40 ); 

    }

    if( $do_y < 31 ) { 

        $$y_ref += .33 if ( $y_plus_minus <= 18 ); 
        $$y_ref -= .33 if ( $y_plus_minus >= 33 ); 
    }
    else { 

        $$y_ref += .33 if ( $y_plus_minus <= 27 ); 
        $$y_ref -= .33 if ( $y_plus_minus >= 40 ); 

    } 

} # END _scatter 

#-----------------------------------------------------------------
# $self->generate_graph( )
#-----------------------------------------------------------------
# This method generates our image graph and returns the entire
# image in a scalar. It takes the following arguments 
#
# xaxis     =>  'EI'
# yaxis     =>  'SN'
# users     =>  '1, 2, 3, 4'        -- List of IDs from user_data table
#
#-----------------------------------------------------------------
sub generate_graph { 
    my $self    =   shift; 
    my $args    =   shift;      # hash reference of names arguments

    # Get our current user's ID 
    my $current_user = $self->auth_user_row->id; 

    my $xaxis = $args->{xaxis}; 
    my $yaxis = $args->{yaxis}; 

    # Ensure both x and y axies are passed as arguments 
    die "No x-axis type specified" if !$xaxis; 
    die "No y-axis type specified" if !$yaxis; 
  
    # Ensure both x and y axies are valid 
    die "Invalid x-axis type '$xaxis'" if $xaxis !~ $valid_axis_type_re; 
    die "Invalid y-axis type '$yaxis'" if $yaxis !~ $valid_axis_type_re; 

    # Ensure we are given some users 
    die "No users specified" if !$args->{users}; 
    die "Invalid user specification" if $args->{users} !~ /,/o; 

    # Build a temporary file name to put this image into due to
    # how oddly gnuplot is building this.  Essentially we can't
    # fall through Gantry to mod_perl's STDOUT in a way that
    # works with gnuplot. 
    my $temp_file = File::Temp::tempnam(    $self->temp_image_dir, 
                                            'search-image'          ); 

    my $dbh = $self->get_schema()->storage->dbh; 

    # Get this user's data 
    my $x_select = $axis_selects->{ $xaxis }; 
    my $y_select = $axis_selects->{ $yaxis }; 

    my $user_sth = $dbh->prepare(
        qq{
            SELECT $x_select, $y_select FROM tt_summary 
            WHERE active = 't'
            AND user_data = $current_user; 
        }
    ); 

    my @this_user; 
    $user_sth->execute; 
    my ($user_x, $user_y) = $user_sth->fetchrow; 
    $user_sth->finish; 

    # Fix bad testing data, but ok to leave in production
    $user_x = 0 if !$user_x; 
    $user_y = 0 if !$user_y; 

    push(@this_user, [ $user_x, $user_y ] ); 

    # Get the rest of the data 
    my $data_sth = $dbh->prepare( 
        qq{ SELECT user_data, active, $x_select, $y_select 
            FROM tt_summary WHERE 
            user_data IN ( $args->{users} ) 
            ORDER BY user_data 
        }
    ); 

    $data_sth->execute; 

    my @in_contact; 
    my @no_contact; 
    my @sent; 
    my @received; 
    my @no_ttt; 

    my %seen; 

    while( my ($user, $status, $x , $y ) = $data_sth->fetchrow ) { 

        $seen{$user}++; 

        my $relationship = $self->_determine_relationship( $user ); 

        # Fix bad testing data, but still valid for production use
        $x = 0 if !$x; 
        $y = 0 if !$y; 

        $self->_scatter( \$x, \$y ); 

        if( $relationship = 'incontact' ) { 
            push( @in_contact, [ $x, $y ] ); 
        }
        elsif( $relationship = 'nocontact' ) { 
            push( @no_contact, [ $x, $y ] ); 
        }
        elsif( $relationship = 'sent' ) { 
            push( @sent, [ $x, $y ] ); 
        }
        elsif( $relationship = 'received' ) { 
            push( @received, [ $x, $y ] ); 
        }
        elsif( $status = 'f' or $status == 0 ) { 
            push( @no_ttt, [ $x, $y ] ); 
        }
        else { 
            die "Unknown relationship"; 
        }
    } 

    foreach my $u ( split(/,/, $args->{users} ) ) { 
        if( !$seen{$u} ) { 
            push( @no_ttt, [ 0, 0 ] ); 
        }
    }

    # Build up gnuplot data 
    my @gnu_args; 

    push( @gnu_args, 
        # Main graph info 
        {
            title           =>  "$xaxis vs. $yaxis", 
            xrange          =>  $graph_ranges->{$xaxis},
            yrange          =>  $graph_ranges->{$yaxis},
            "output file"   =>  $temp_file, 
            "output type"   =>  "png", 
        }, 
        # This user's data 
        [
            {
                title       =>  'Self', 
                type        =>  'matrix', 
            }, 
            \@this_user,
         ] 
    ); 

    # In contact
    if( @in_contact ) { 

        push( @gnu_args, 
            [
                {
                    title       =>  'In Contact', 
                    type        =>  'matrix',
                },
                \@in_contact, 
            ] 
        );
    }

    if( @no_ttt ) { 
        push( @gnu_args, 
            [
                {
                    title       =>  'No TTT Taken', 
                    type        =>  'matrix',
                },
                \@no_ttt, 
            ], 
        );
    }

    # No contact
    if( @no_contact ) { 
        
        push( @gnu_args, 
            [
                {
                    title       =>  'No Contact', 
                    type        =>  'matrix',
                },
                \@no_contact, 
            ], 
        );
    }

    # Sent
    if( @sent ) { 

       push( @gnu_args, 
            [
                {
                    title       =>  'Sent',
                    type        =>  'matrix',
                },
                \@sent, 
            ] 
        ); 
    }

    # Received
    if( @received ) { 
        push( @gnu_args, 
            # Received
            [
                {
                    title       =>  'Received',
                    type        =>  'matrix',
                },
                \@received, 
            ] 
        ); 
    }

    use Data::Dumper;
    warn( Dumper( \@gnu_args ) ); 

    gnuplot( @gnu_args ); 

    # Now that our image is built in our temp file, read it 
    # back in and send it to the browser
    my $image = read_file( $temp_file ); 

    # Remove the file so we don't eat up disk space 
    unlink( $temp_file ); 

    return( $image ); 

} # END generate_graph 

#-----------------------------------------------------------------
# $self->perform_search( )
#-----------------------------------------------------------------
# This method takes in the various search parameters ( same
# as in search.tt ) and returns either the information necessary
# for the SOAP request or the values necessary for the HTML
# display 
#
# Parameters: 
#
#   limit               --  Number of rows to return 
#   offset              --  Page number 
#
#   no_contact          --  Not already a contact 
#   no_request          --  Have not sent a request to this user 
#   (not done) new_since_logout    --  New users since last time I logged out 
#   life_motto_contains --  
#   ttt_types           --  I*T* etc. 
#-----------------------------------------------------------------
sub perform_search { 
    my $self        =   shift; 
    my $args        =   shift;  # Hash ref of search options 

#    use Data::Dumper; 
#    warn( Dumper( \$args ) ); 

    # Get our current user's ID 
    my $current_user = $self->auth_user_row->id; 

    # Get our schema 
    my $s = $self->get_schema; 

    # Setup our LIMIT and OFFSET for paging 
    my $limit       = $args->{limit}; 
    my $offset      = $args->{offset}; 

    if( !$limit or $limit !~ /^\d+$/o ) { 
        $limit = $self->fish_config('search_page_item_limit'); 
    }

    if( !$offset or $offset !~ /^\d+$/o ) { 
        $offset = 0; 
    }

    # Variables to store the parts of the SQL we're building based
    # on the search criteria 
    my @selects; 
    my @where_parts; 
    my %froms; 

    # We'll always be selecting from this table 
    $froms{user_data} = 1; 

    # Setup selects based on SOAP vs HTML 
    my $type = $args->{type};
    if( $type eq 'SOAP' ) { 
        @selects = (
                        'user_data.id as id', 
                        'user_data.user_name as user_name', 
                        'user_data.full_name as full_name',
                        'user_data.email as email',
                        'user_data.town as town',
                        'user_data.country as country',
                        'user_data.life_motto as life_motto',
                        'user_data.secondary_life_motto as secondary_life_motto',
                        'user_data.GPS_latitude as latitude',
                        'user_data.GPS_longitude as longitude',
                        'user_data.enne_primary as enne_primary',
                        'user_data.enne_secondary as ene_secondary',
                        qq{
                            ( SELECT short_summary FROM tt_summary
                              WHERE tt_summary.user_data = user_data.id
                              AND tt_summary.active = 't') as ttt_type
                        }, 
        ); 

    }
    elsif( $type eq 'HTML' ) { 

        @selects = ( 
                        'user_data.id as id', 
                        'user_data.user_name as user_name',
                        'user_data.town as town',
                        'user_data.country as country',
                        'user_data.life_motto as life_motto',
                        'user_data.secondary_life_motto as secondary_life_motto',
                        'user_data.enne_primary as enne_primary',
                        'user_data.enne_secondary as enne_secondary',
                        qq{
                            ( SELECT short_summary FROM tt_summary
                              WHERE tt_summary.user_data = user_data.id
                              AND tt_summary.active = 't') as ttt_type
                        }, 
        ); 

    }
    else { 
        die "Invalid search type '$type'"; 
    }

    my $some_search = 0; 

    # Deal with contact/request search parameters 
    if( $args->{no_contact} and !$args->{no_request} ) { 
        
        push( @where_parts,  
               qq{ ( 
                user_data.id NOT IN ( 
                    select contact_id FROM contacts WHERE 
                    contacts.user_id = 9 )
                  ) } 
        ); 

        $froms{'contacts as c'} = 1; 

        $some_search = 1; 
    }
    # This actually handles both the case of no_request and
    # no_request + no_contact 
    elsif( $args->{no_request} ) { 
        push( @where_parts, 
                qq{ ( 
                       visible.user_rec = user_data.id AND
                       visible.user_send <> $current_user AND 
                      ( status = 1 OR status = 2 ) ) } ); 

        $froms{visible} = 1; 
        $some_search = 1; 
    }

    # Handle life motto 
    if( $args->{life_motto_contains} ) { 
       
        my $s = $s->storage->dbh->quote( "\%%$args->{life_motto_contains}\%" ); 

        push( @where_parts, 
            qq{
                ( 
                   user_data.life_motto LIKE $s OR 
                   user_data.secondary_life_motto LIKE $s
                )
            }
        ); 
        $some_search = 1; 

    }

    # Deal with TTT types searches 
    if( $args->{ttt_types} and $args->{ttt_types} ne '****' ) { 

        $args->{ttt_types} =~ s/\*/\_/og; 
        
        push( @where_parts, 
            qq{ 
                ( user_data.id IN
                    ( SELECT DISTINCT user_data FROM tt_summary 
                        WHERE 
                        active = 't' AND 
                        short_summary LIKE '$args->{ttt_types}' ) 
                )
            }
        ); 

        $some_search = 1; 
    }

    # Deal with new since 
    #if( $args->{new_since_logout} ) { 
    #    push( @where_parts, 
    #            qq{ ( ul.created > user_data.last_logout
    #                  AND user_data.id = $current_user ) } ); 
    #
    #    $froms{'user_data as ul'} = 1
    #} 

    # Build our SQL 
    my $sql = 'SELECT ' . join( ', ', @selects ) . 
              ' FROM ' . join( ', ', keys( %froms ) ) . 
              ' WHERE ' . join( ' AND ', @where_parts ) . 
              ' LIMIT ? OFFSET ?'; 

    if( !$some_search ) { 
        return( { 
                   no_search => 1 
               } ); 
    }

    warn( $sql ); 

    # Attempt to build our query 
    my $dbh = $s->storage->dbh; 
    my $sth = $dbh->prepare( $sql ) or
        die "Cannot prepare search sql: $!"; 

    # Execute with this page's limit/offset 
    $sth->execute( $limit, $offset ) or 
        die "Cannot execute search sql: $!";

    # Return a hash ref by ID 
    my $data_ref = $sth->fetchall_hashref( 'id' )
        or die "Cannot fetchall: $!"; 

    warn( Dumper( \$data_ref ) ); 

    $sth->finish; 

    return( $data_ref ); 

} # END perform_search 

#-----------------------------------------------------------------
# $self->do_graph( )
#-----------------------------------------------------------------
# This method generates our graph based on the search parameters
# the user entered in from the do_main() page 
#-----------------------------------------------------------------
sub do_graph { 
    my $self    =   shift; 

    # Get our input parameters 
    my $ins = $self->params; 

    # Change our content type to be and disable templating since
    # we are building an image
    $self->content_type( 'image/png' ); 
    $self->template_disable( 1 ); 

    # Build our image 
    my $image = $self->generate_graph( {
                                            xaxis   => $ins->{xaxis}, 
                                            yaxis   => $ins->{yaxis}, 
                                            users   => $ins->{users}, 
                                        } ); 
                    
    return( $image ); 

} # END do_graph 

1;

=head1 NAME

PbT::Search - A search controller in the PbT application

=head1 SYNOPSIS

This package is meant to be used in a stand alone server/CGI script or the
Perl block of an httpd.conf file.

Stand Alone Server or CGI script:

    use PbT::Search;

    my $cgi = Gantry::Engine::CGI->new( {
        config => {
            #...
        },
        locations => {
            '/search' => 'PbT::Search',
            #...
        },
    } );

httpd.conf:

    <Perl>
        # ...
        use PbT::Search;
    </Perl>

    <Location /search>
        SetHandler  perl-script
        PerlHandler PbT::Search
    </Location>

If all went well, one of these was correctly written during app generation.

=head1 DESCRIPTION

Provides the user search features of the PbT application.

=head1 METHODS

=over 4

=item do_search

=back


=head1 DEPENDENCIES

    PbT
    Gantry::Utils::CRUDHelp

=head1 AUTHOR

Phil Crow, E<lt>phil@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2007 Phil Crow

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut
