<?php

/**
 * Base class for Active record manager
 * The instance of this class is normally associated with one database table record
 * @package core
 */
class pfBase {

	/**
	 * $sMainTable indicates main record table
	 *
	 * @var string
	 */
	public $sMainTable = null;

	/**
	 * Unique object id.
	 * When record data is loaded $sId is normally associated with 'id' field value
	 *
	 * @var string
	 */
	public $sId;

	/**
	 * Array containing table metadata and field descriptions
	 *
	 * @var array
	 */
	private $aMetaData = null;

    /**
     * $aFields array contains field obejcts (including field values)
     *
     * @var array
     */
	public $aFields = null;

    /**
	 * Required DB fields
	 *
	 * @var array
	 */
	protected $_aRequiredFields = array();

	/**
	 * Non editable fields
	 *
	 * @var array
	 */
	protected $_aProtectedFields = array();

	/**
	 * Object initialisation.
	 *
	 */
	public function __construct()
	{

	    if (isset($this->sMainTable))
	       $this->Init($this->sMainTable);
	}

	/**
	 * Sets table names and loads table meta data
	 *
	 * @param string $sTable
	 */
	public function init($sTable)
	{

	    //reseting object values
	    $this->aMetaData = null;
	    $this->aFields = null;
	    $this->sMainTable = null;
	    $this->sId = null;

	    if (!$sTable)
	       return;

	    $this->sMainTable = $sTable;

	    //todo: add some data caching to improve performance
	    //when we have problems with it
        $oDB = pfDB::getDB();

        $this->aMetaData = $oDB->describeTable($sTable);

        //init object properties according to table meta data
        foreach ($this->aMetaData as $aField) {
            $sFieldName = strtolower($aField['name']);
            $this->aFields[$sFieldName] = & pfUtils::pfNew('pfField');
            $this->aFields[$sFieldName]->sName = $sFieldName;
        }
	}

	/**
	 * Assigns $aInput array values (supposedly from user input) to current instance.
	 * In case $blOverwriteEmpty is set to true, not set object properties are set to blank, and left intact otherwise
	 *
	 * @param array $aInput
	 * @param bool $blOverwriteEmpty
	 * @return unknown
	 */
	public function assign($aInput, $blOverwriteEmpty = true)
	{
	    if (!$this->aMetaData || !$this->aFields)
	       throw new Exception('Data assignment exception');

	    $blIsSet = false;
        foreach ($this->aMetaData as $aField) {
            $sFieldName = strtolower($aField['name']);
            if (isset($aInput[$sFieldName])) {
                $this->aFields[$sFieldName]->setValue($aInput[$sFieldName]);
                $blIsSet = true;

                //setting id
                if ($sFieldName == 'id')
                    $this->sId = $aInput[$sFieldName];
            }
            elseif ($blOverwriteEmpty)
                $this->aFields[$sFieldName]->setValue(null);
        }

        return $blIsSet;
	}

	/**
	 * Loads record data from database table by $sId and assigns to current object
	 *
	 * @param unknown_type $sId
	 */
	public function load($sId)
	{
	    if (!$sId || !$this->aFields)
	       throw new Exception('Data load exception');

	    $oDB = pfDB::getDB();

	    $sQ = $oDB->quoteInto('select * from ' . $this->sMainTable . ' where id = ?', $sId);
	    $oResult = $oDB->query($sQ);
	    $aRow = $oResult->fetchAll();
	    if (isset($aRow[0]))
	    {
	       foreach ($aRow[0] as $sKey => $sVal) {
	           $sFieldName = $sKey;
	           $this->aFields[$sFieldName]->setValue($sVal);
	       }
	       $this->sId = $sId;
	       return true;
	    }

	    $this->init($this->sMainTable);
	    return false;
	}

	/**
	 * Saves current object data to the database
	 *
	 */
	public function save()
	{
	    if (!$this->sMainTable)
	       throw new Exception('DB write exception');

        if ($this->exists())
            return $this->_update();
        else
            return $this->_insert();
	}

	/**
	 * Deletes current record from db
	 *
	 */
	public function delete()
	{
        if (!$this->sId || !$this->sMainTable)
	       throw new Exception('Delete exception');

	    $oDB = pfDB::getDB();
	    $sWhere = $oDB->quoteInto('id = ?', $this->sId);
	    $oDB->delete($this->sMainTable, $sWhere);
	    $this->init($this->sMainTable);
	    return true;
	}

	/**
	 * Checks if record having id $sId exists in database table.
	 *
	 * @param string $sId
	 * @return bool true if exists false otherwise.
	 */
	private function exists()
	{

	    if (!$this->sMainTable)
	       throw new Exception('DB write exception');

	    if (!$this->sId)
	       return false;

	    $oDB = pfDB::getDB();
	    $sQ = $oDB->quoteInto('select id from `' . $this->sMainTable . '` where id = ?', $this->sId);
	    $oResult = $oDB->query($sQ);
	    $aRows = $oResult->fetchAll();

	    if (count($aRows) > 0)
	       return true;
        else
	       return false;
	}

	/**
	 * Inserts new record.
	 *
	 * @return int Last inserted id or false on failure.
	 */
	protected function _insert()
	{
        if (!$this->sMainTable || !$this->aMetaData)
	       throw new Exception('DB write exception');

	    //checking required fields
	    foreach ($this->_aRequiredFields as $sField)
	       if (!$this->aFields[$sField]->sValue)
	           throw new Exception('Please supply all required fields');

	    $aInsertFields = array();
	    foreach ($this->aMetaData as $aField)
        {
            $sFieldName = strtolower($aField['name']);
            //if ($sFieldName != 'id')
            if(isset($this->aFields[$sFieldName]->sValue))
                $aInsertFields[$sFieldName] = $this->aFields[$sFieldName]->sValue;
        }

        $oDB = pfDB::getDB();
        $iRowsAffected = $oDB->insert($this->sMainTable, $aInsertFields);

        if ($iRowsAffected)
        {
            $this->sId = $oDB->lastInsertId();
            $this->aFields['id']->setValue($this->sId);
            return $this->sId;
        }
        else
            throw new Exception('DB write exception');
	}

	/**
	 * Updates existing record.
	 *
	 * @return int Updated id or false on failure.
	 */
	protected function _update()
	{

        if (!$this->sId || !$this->sMainTable || !$this->aMetaData)
	       throw new Exception('DB write exception');

	    $aUpdateFields = array();
	    foreach ($this->aMetaData as $aField)
        {
            $sFieldName = strtolower($aField['name']);
            if ($sFieldName != 'id' &&
                isset($this->aFields[$sFieldName]->sValue) &&
                !in_array($sFieldName, $this->_aProtectedFields)) {

                $aUpdateFields[$sFieldName] = $this->aFields[$sFieldName]->sValue;
                //checking required fields
                if (!$this->aFields[$sFieldName]->sValue && in_array($sFieldName, $this->_aRequiredFields))
                    throw new Exception('Please supply all required fields');
            }


        }

        $oDB = pfDB::getDB();

        $sWhere = $oDB->quoteInto('id = ?', $this->sId);

        $oDB->update($this->sMainTable, $aUpdateFields, $sWhere);

        return $this->sId;
	}

}