<?php

/**
 * @author 	   Michael Mayer <michael.mayer@zend.com>
 */

/** MultilangModel */
require_once('models/MultilangModel.php');

/** D100_Container */
require_once('D100/Container.php');

/**
 * Multi-Lang Content Model
 */
abstract class MultilangContentModel extends MultilangModel {
  /**
   * This multi-dimensional array holds the field values for this content model
   *
   * @var array
   */
  protected $_content = array();
  /**
   * Original content, so that you can calculate a diff for changed data
   *
   * @var array
   */
  protected $_originalContent = array();
  /**
   * An optional query filter
   *
   * @var array
   */
  protected $_filter = array();
  /**
   * Was the content data already loaded?
   *
   * @var boolean
   */
  protected $_isLoaded = false;
  /**
   * Was the content data changed?
   *
   * @var boolean
   */
  protected $_isChanged = false;
  /**
   * Which fields were changed at all?
   *
   * @var array
   */
  protected $_changedFields = array();
  /**
   * Fallback to the default locale if current locale doesn't exist?
   *
   * @var boolean
   */
  protected $_mergeDefaultLocale = true;
  /**
   * The default locale
   *
   * @var string
   */
  protected $_defaultLocale = DEFAULT_LOCALE;
  /**
   * The primary ID of this content object
   *
   * @var integer
   */
  protected $_id = null;

  /**
   * Returns an instance of this MultilangContentModel
   *
   * @param integer $id
   * @param string $locale
   * @param array $filter
   * @return MultilangContentModel
   */
  public abstract static function getInstance ($id = null, $locale = null, $filter = null);

  /**
   * Protected funczion to be used by the getInstance function of inherited models
   *
   * @param string $className
   * @param integer $id
   * @param string $locale
   * @param array $filter
   * @return MultilangContentModel Well, not really, it's an instance of $className
   */
  protected static function getMultilangContentInstance ($className, $id = null, $locale = null, $filter = null) {
    $result = null;

    if(empty($className)) {
      throw new Exception('$className must not be empty in MultilangContentModel::getInstance()');
    }

    if(!is_null($id) && !is_numeric($id)) {
      throw new Exception('$id must be null or numeric in '.$className.'::getInstance ()');
    }

    if($id != null) {
      $registryName = $className.'_'.$id;

      if(Zend::isRegistered($registryName)) {
        $result = Zend::registry($registryName);
      }
      else {
        $result = new $className ($id, $locale, $filter);
        if($result->isLoaded) {
          Zend::register($registryName, $result);
        }
      }
    }
    else {
      $result = new $className (null, $locale, $filter);
    }

    return $result;
  }

  /**
   * The contructor already loads the field values from the database, if a ID is given
   *
   * @param integer $id
   * @param sring $locale
   * @param array $filter Optional
   */
  protected function __construct($id = null, $locale = null, $filter = null) {
    parent::__construct($locale);

    if($filter != null) {
      $this->_filter = $filter;
    }

    if(!empty($id)) {
      $this->load($id);
    }
  }

  public function getId () {
    return $this->_id;
  }

  /**
   * Load the content from the database
   *
   * @param integer $id
   * @return boolean
   * @todo Make more use of the Zend Registry
   */
  public function load ($id = null) {
    $db = $this->getDb();

    $result = false;

    if($id == null) {
      if($this->_isLoaded && $this->_id != null) {
        $id = $this->_id;
      }
      else {
        throw new Exception('id parameter is required to load content object');
      }
    }

    $select = $db->select();

    $select->from($this->_tableName, '*');
    $select->where($this->_tableKey.' = ?', $id);

    // Add filter
    if(is_array($this->_filter)) {
      foreach($this->_filter as $filterKey => $filterValue) {
        if($filterKey != null && $filterValue != null) {
          $select->where($filterKey.' = ?', $filterValue);
        }
      }
    }

    $results = $db->fetchAll($select);

    if(count($results) == 0) {
      throw new Exception('Content object (id '.$id.') does not seem to exist in database');
    }

    // Store results to $this->_content
    if($this->_tableLocale != '') {
      foreach($results as $result) {
        $this->_content[$result[$this->_tableLocale]] = array();

        foreach($result as $key => $value) {
          $this->_content[$result[$this->_tableLocale]][$key] = $value;
        }
      }
    }
    elseif(count($results) == 1) {
      $this->_content[$this->_defaultLocale] = current($results);
    }
    else {
      throw new Exception('No tableLocale but more than one result for this content object (id '.$id.')');
    }

    if(!isset($this->_content[$this->_defaultLocale])) {
      throw new Exception('Default locale doesn\'t seem to exist for this content object (id '.$id.')');
    }

    $this->_isLoaded = true;
    $this->_isChanged = false;
    $this->_id = $id;

    $this->_originalContent = $this->_content;

    return true;
  }

  /**
   * Save all data of this content model to the database
   *
   * @return boolean
   */
  public function save () {
    if($this->_isChanged == false) { // Take the short way, if nothing was changed
      return false;
    }

    $db = $this->getDb();
    $log = $this->getLog();

    $logId = $log->getLogId();

    if($this->_isLoaded && is_numeric($this->_id)) {
      $id = $this->_id;
      try {
        foreach($this->_content as $locale => $localeContent) {
          $select = $db->select();

          $select->from($this->_tableName, 'COUNT(*) as count');
          $select->where($db->quoteIdentifier($this->_tableKey).' = ?', $id);

          if($this->_tableLocale != '') {
            $select->where($db->quoteIdentifier($this->_tableLocale).' = ?', $locale);
          }

          $rowCount[0] = $db->fetchOne($select);

          if($rowCount[0] == 1) { // Update row, if locale exists
            $update = array();
            $oldValues = array();

            if(isset($this->_changedFields[$locale])) {

              foreach($this->_changedFields[$locale] as $fieldName => $fieldVal) {
                $update[$fieldName] = $localeContent[$fieldName];
                if(isset($this->_originalContent[$locale][$fieldName])) {
                  $oldValues[$fieldName] = $this->_originalContent[$locale][$fieldName];
                }
              }

              $where = $db->quoteInto($this->_tableKey.' = ? ', $id);

              if($this->_tableLocale != '') {
                $where .= ' AND '.$db->quoteInto($this->_tableLocale.' = ? ', $locale);
              }

              $db->update($this->_tableName, $update, $where);
              $log->addEvent('UPDATE', $this->_tableName, $this->_tableKey, $id, $locale, $oldValues, $update);
            }
          }
          elseif($rowCount[0] == 0) { // Insert new row, if locale does not exist yet
            $localeContent[$this->_tableKey] = $this->_id;

            if($this->_tableLocale != '') {
              $localeContent[$this->_tableLocale] = $locale;
            }

            $db->insert($this->_tableName, $localeContent);
            $log->addEvent('INSERT', $this->_tableName, $this->_tableKey, $this->_id, $locale, null, $localeContent);
          }
          else { // Throw error, if locale exists more than once
            throw new Exception('There were too many rows for locale "'.$locale.'" while saving content object (id '.$id.')');
          }
        }
        // $db->commit();
      }
      catch(Exception $e) {
        // $db->rollBack();
        $log->addEvent('ERROR', $this->_tableName, $this->_tableKey, $this->_id, $this->getLocale(), null, null, $e->getMessage());
        throw new Exception('Error while saving data of content object (id '.$id.'): '.$e->getMessage());
      }
    }
    else {
      // Create new id
      $select = $db->select();

      $select->from($this->_tableName, 'max('.$this->_tableKey.') + 1');

      $newId[0] = $db->fetchOne($select);

      if(isset($newId[0]) && is_numeric($newId[0])) {
        $this->_id = $newId[0];

        foreach($this->_content as $locale => $localeContent) {
          if(isset($localeContent[$this->_tableKey]) && !empty($localeContent[$this->_tableKey])) {
            $this->_id = $localeContent[$this->_tableKey];
          }
          else {
            $localeContent[$this->_tableKey] = $this->_id;
          }

          if($this->_tableLocale != '') {
            $localeContent[$this->_tableLocale] = $locale;
          }

          $log->addEvent('INSERT', $this->_tableName, $this->_tableKey, $this->_id, $locale, null, $localeContent);
          $db->insert($this->_tableName, $localeContent);
        }

        Zend::register($this->className.'_'.$this->_id, $this);
      }
    }

    return true;
  }

  /**
   * Delete content model data from database
   *
   * @return boolean
   */
  public function delete () {
    if($this->_isLoaded && $this->_id != null) {
      $id = $this->_id;
    }
    else {
      throw new Exception('Content object (id '.$id.') can\'t be deleted because it was not successfully loaded');
    }

    $db = $this->getDb();
    $log = $this->getLog();

    $logId = $log->getLogId();

    $where = $db->quoteInto($db->quoteIdentifier($this->_tableKey).' = ?', $id);

    try {
      $db->delete($this->_tableName, $where);
      $log->addEvent('DELETE', $this->_tableName, $this->_tableKey, $id, $this->getLocale());
    } catch (Exception $e) {
      $error = 'Content object (id '.$id.') was not deleted - 0 rows affected by delete query';
      $log->addEvent('ERROR', $this->_tableName, $this->_tableKey, $this->_id, $this->getLocale(), null, null, $error);
      throw new Exception($error);
    }

    return true;
  }

  /**
   * Wrapper for setValue()
   *
   * @param string $name
   * @param mixed $value
   */
  private function __set ($name, $value) {
    $this->setValue($name, $value);
  }

  /**
   * Wrapper for getValue()
   *
   * @param string $name
   * @return mixed
   */
  private function __get ($name) {
    return $this->getValue($name);
  }

  /**
   * Checks if a value is set for the current OR the default locale
   *
   * @param string $name
   * @return boolean
   */
  private function __isset ($name) {
    $result = isset($this->_content[$this->getLocale()][$name]) || isset($this->_content[$this->_defaultLocale][$name]);

    return $result;
  }

   /**
    * Unset a field value for the current locale
    *
    * @param string $name
    */
  private function __unset ($name) {
    // Unset always affects the current language only!

    if(!isset($this->_content[$this->getLocale()])) {
      throw new Exception('Can\'t unset '.$name.' because locale '.$this->getLocale().' does not exist in __unset()');
    }

    if(isset($this->_content[$this->getLocale()][$name])) {
      unset($this->_content[$this->getLocale()][$name]);
    }
  }

  /**
   * Get all field values for the given locale.
   * Set the third parameter to false, if you don't want a fallback to the default locale if a field's value is NULL.
   *
   * @param string $locale
   * @param boolean $mergeDefaultLocale
   * @return string
   */
  public function getValues ($locale = null, $mergeDefaultLocale = null) {
    $result = array();

    if($locale == null) {
      $locale = $this->getLocale();
    }

    if($mergeDefaultLocale == null) {
      $mergeDefaultLocale = $this->_mergeDefaultLocale;
    }

    if(empty($locale)) {
      throw new Exception('No locale defined in getValues()');
    }

    if(isset($this->_content[$locale])) {
      foreach($this->_content[$locale] as $key => $value) {
        $result[$key] = $value;
      }
    }

    if($mergeDefaultLocale && $this->_defaultLocale != $locale && isset($this->_content[$this->_defaultLocale])) {
      foreach($this->_content[$this->_defaultLocale] as $key => $value) {
        if(!isset($result[$key]) || $result[$key] == null) {
          $result[$key] = $value;
        }
      }
    }

    return $result;
  }

  /**
   * Get field value for the given locale. Set the third parameter to false, if you don't want a fallback to the default
   * locale (normally 'en')
   *
   * @param string $name
   * @param string $locale
   * @param boolean $mergeDefaultLocale
   * @return mixed
   */
  public function getValue ($name, $locale = null, $mergeDefaultLocale = null) {
    $result = null;

    if($locale == null) {
      $locale = $this->getLocale();
    }

    if($mergeDefaultLocale == null) {
      $mergeDefaultLocale = $this->_mergeDefaultLocale;
    }

    if(empty($locale)) {
      throw new Exception('No locale defined in getValue()');
    }

    // Check if content is available in current language
    if(isset($this->_content[$locale][$name])) {
      $result = $this->_content[$locale][$name];
    }
    // If not, check default language
    elseif($mergeDefaultLocale && $this->_defaultLocale != $locale && isset($this->_content[$this->_defaultLocale][$name])) {
      $result = $this->_content[$this->_defaultLocale][$name];
    }

    return $result;
  }

  /**
   * Set field value for the given locale
   *
   * @param string $name
   * @param mixed $value
   * @param string $locale
   */
  public function setValue ($name, $value, $locale = null) {
    if(empty($name)) {
      throw new Exception('Empty name in setValue is not allowed');
    }

    if($locale == null) {
      $locale = $this->getLocale();
    }

    if(strlen($locale) > 2 && strpos($locale, '_') === false) {
      throw new Exception('The locale fomat is not correct: '.$locale);
    }

    if(empty($locale)) {
      throw new Exception('No locale defined in setValue()');
    }

    if(!isset($this->_content[$locale])) {
      $this->_content[$locale] = array();
    }

    if(!isset($this->_content[$locale][$name]) || $this->_content[$locale][$name] != $value) {
      $this->_changedFields[$locale][$name] = true;
    }

    $this->_content[$locale][$name] = $value;

    $this->_isChanged = true;
  }

  /**
   * Was the data for this content model already loaded from the database
   *
   * @return bollean
   */
  public function isLoaded () {
    return ($this->_isLoaded == true);
  }

  /**
   * Was the data of this content model changed?
   *
   * @return boolean
   */
  public function isChanged () {
    return ($this->_isLoaded == true);
  }

  /**
   * Fill a D100_Container with content from this object and locale
   *
   * @param D100_Container $container
   * @param string $locale
   */
  public function fillContainer (D100_Container $container = null, $locale = null) {
    $def = $container->getDef();

    if($locale == null) {
      $locale = $this->getLocale();
    }

    foreach($def->getDeclaredFields() as $name) {
      $fieldDef = $def->getFieldDef($name);

      if(isset($fieldDef['defaultlocale']) && $fieldDef['defaultlocale'] == true && isset($this->_content[$this->_defaultLocale][$name])) {
        $container->$name = $this->_content[$this->_defaultLocale][$name];
      }
      elseif(isset($this->_content[$locale][$name]) && isset($this->_content[$locale][$name])) {
        $container->$name = $this->_content[$locale][$name];
      }
      else {
        $container->$name = null;
      }
    }
  }

  /**
   * Assing the data of a D100_Container (value object) to this content model
   *
   * @param D100_Container $container
   * @param string $locale
   */
  public function assignContainer (D100_Container $container, $locale = null) {
    $def = $container->getDef();

    if($locale == null) {
      $locale = $this->getLocale();
    }

    $this->_isChanged = true;

    foreach($def->getDeclaredFields() as $name) {
      $fieldDef = $def->getFieldDef($name);

      if(isset($fieldDef['defaultlocale']) && $fieldDef['defaultlocale'] == true) {
        $this->_content[$this->_defaultLocale][$name] = $container->$name;
        $this->_changedFields[$this->_defaultLocale][$name] = true;
      }
      elseif(empty($container->$name)) {
        if(isset($this->_content[$locale][$name])) {
          $this->_changedFields[$locale][$name] = true;
          $this->_content[$locale][$name] = null;
        }
      }
      else {

        if(!isset($this->_content[$locale]) || $this->_content[$locale][$name] != $container->$name) {
          $this->_changedFields[$locale][$name] = true;
        }

        if(empty($this->_content[$this->_defaultLocale][$name])) {
          $this->_content[$this->_defaultLocale][$name] = $container->$name;
          $this->_changedFields[$this->_defaultLocale][$name] = true;
        }

        $this->_content[$locale][$name] = $container->$name;
      }
    }
  }

  /**
   * Do a many-to-many multi-locale relation query :-)
   *
   * @param string $relTable
   * @param string $dataTable
   * @param string $joinOn The key to join on, e.g. userId
   * @param string $where
   * @param array $fields
   * @param string $sortOrder
   * @param array $extraJoins
   * @return array A multi-dimensional array
   */
  protected function getMultilangRelations ($relTable, $dataTable, $joinOn, $where, Array $fields, $sortOrder = null, $extraJoins = null) {
    $results = array();

    if(!$this->_isLoaded) {
      return $results;
    }

    if(!in_array($joinOn, $fields)) {
      $fields[] = 'a.'.$joinOn;
    }

    $db = $this->getDb();

    $select = $db->select();

    $select->from($relTable.' a', $fields);
    $select->join($dataTable.' b', 'b.'.$joinOn.' = a.'.$joinOn);

    if(is_array($extraJoins) && count($extraJoins) > 0) {
      $chr = 99; // 'c'
      foreach($extraJoins as $joinTable => $joinJoin) {
        $select->join($joinTable.' '.chr($chr++), $joinJoin);
      }
    }

    $select->where($where);
    $select->where('b.locale = ?', DEFAULT_LOCALE);
    if($sortOrder != null) {
      $select->order($sortOrder);
    }

    // Fetch data for DEFAULT locale
    $defaultResults = $db->fetchAll($select);

    foreach($defaultResults as $result) {
      $results[$result[$joinOn]] = $result;
    }

    // Fetch data for current locale, if it's different from the DEFAULT locale
    if($this->getLocale() != null && $this->getLocale() != DEFAULT_LOCALE) {
      $select = $db->select();

      $select->from($relTable.' a', $fields);
      $select->join($dataTable.' b', 'b.'.$joinOn.' = a.'.$joinOn);

      if(is_array($extraJoins) && count($extraJoins) > 0) {
        $chr = 99; // 'c'
        foreach($extraJoins as $joinTable => $joinJoin) {
          $select->join($joinTable.' '.chr($chr++), $joinJoin);
        }
      }

      $select->where($where);
      $select->where('b.locale = ?', $this->getLocale());

      $defaultResults = $db->fetchAll($select);

      foreach($defaultResults as $result) {
        foreach($result as $key => $val) {
          if(!empty($val)) {
            $results[$result[$joinOn]][$key] = $val;
          }
        }
      }

    }

    return $results;
  }

  /**
   * Set many-to-many relation
   *
   * @param integer $relId
   * @param string $relKey
   * @param string $relTable
   * @param array $options
   */
  public function setRelation($relId, $relKey, $relTable, $options = array()) {
    if(!is_numeric($relId) || $relId < 1) {
      throw new Exception('relId was not int or < 1 in MultilangContentModel::setRelation()');
    }

    $db = $this->getDb();
    $log = $this->getLog();

    $select = $db->select();

    $select->from($relTable, '*');
    $select->where($this->_tableKey.' = ?', $this->_id);
    $select->where($relKey.' = ?', $relId);

    $row = $db->fetchRow($select);

    $logId = $log->getLogId();

    if(!is_array($row) || count($row) == 0) {
      $options[$this->_tableKey] = $this->_id;
      $options[$relKey] = $relId;
      $db->insert($relTable, $options);
      $log->addEvent('INSERT', $relTable, $this->_tableKey, $this->_id, null, array(), $options, null, $relKey, $relId);
    }
    elseif(is_array($options) && count($options) > 0) {
      $where = $db->quoteInto($this->_tableKey.' = ?', $this->_id);
      $where .= ' AND '.$db->quoteInto($relKey.' = ?', $relId);
      $db->update($relTable, $options, $where);

      $select = $db->select();

      $select->from($relTable, '*');
      $select->where($this->_tableKey.' = ?', $this->_id);
      $select->where($relKey.' = ?', $relId);

      $newRow = $db->fetchRow($select);
      $log->addEvent('UPDATE', $relTable, $this->_tableKey, $this->_id, null, $row, $newRow, '', $relKey, $relId);
    }
  }

  /**
   * Unset a many-to-many relation
   *
   * @param integer $relId
   * @param string $relKey
   * @param string $relTable
   */
  protected function unsetRelation ($relId, $relKey, $relTable) {
    if(!is_numeric($relId) || $relId < 1 || empty($relTable) || empty($relKey)) {
      throw new Exception('relId was not int or < 1 in MultilangContentModel::setRelation()');
    }

    $db = $this->getDb();
    $log = $this->getLog();

    $logId = $log->getLogId();

    $select = $db->select();

    $select->from($relTable, '*');
    $select->where($this->_tableKey.' = ?', $this->_id);
    $select->where($relKey.' = ?', $relId);

    $row = $db->fetchRow($select);

    if(is_array($row)) { // Check for existence
      $where = $db->quoteInto($this->_tableKey.' = ?', $this->_id);
      $where .= ' AND '.$db->quoteInto($relKey.' = ?', $relId);

      try {
        $db->delete($relTable, $where);
        $log->addEvent('DELETE', $relTable, $this->_tableKey, $this->_id, null, $row, array(), '', $relKey, $relId);
      } catch (Exception $e) {
        $log->addEvent('ERROR_DELETE', $relTable, $this->_tableKey, $this->_id, null, $row, array(), '', $relKey, $relId);
      }
    }
  }

  /**
   * Get all the locales defined for this content object
   *
   * @return array
   */
  public function getLocales () {
    return array_keys($this->_content);
  }

}
?>