<?php

require_once('models/MultilangContentModel.php');

class UserModel extends MultilangContentModel {
  /**
   * Returns an intance of UserModel with the given ID ($id is null, if new user)
   *
   * @param integer $id
   * @param string $locale
   * @param array $filter
   * @return UserModel
   */
  public static function getInstance ($id = null, $locale = DEFAULT_LOCALE, $filter = null) {
    return self::getMultilangContentInstance(__CLASS__, $id, DEFAULT_LOCALE);
  }

  /**
   * Returns an intance of UserModel with the given initial (this method also uses the alias table)
   *
   * @param string $initial
   * @param string $locale
   * @param array $filter
   * @return UserModel
   */
  public static function getInstanceByUsername ($username = null) {
    $db = MultilangModel::getDbObject();
    $select = $db->select();

    $select->from('users', 'user_id');
    $select->where('user_name = ?', $username);

    $result = $db->fetchOne($select);

    if(is_numeric($result)) {
      try {
        return self::getMultilangContentInstance(__CLASS__, $result, DEFAULT_LOCALE);
      } catch (Exception $e) {
        return null;
      }
    } else {
      return null;
    }
  }

  protected function __construct($id = null, $locale = DEFAULT_LOCALE) {
    $this->_tableName = 'users';
    $this->_tableKey  = 'user_id';
    $this->_tableLocale = 'locale';

    parent::__construct($id, DEFAULT_LOCALE);
  }

  public function getRCDUsers($status) {
    switch($status) {
      case RCD_in_contact:
        return $this->getRCDUsersInContact();
      case RCD_sent:
        return $this->getRCDUsersSent($status);
      case RCD_received:
        return $this->getRCDUsersReceived($status);
      default:
        throw new Exception("Fatal: status contains invalid value [$status]");
    }
  }

  public function getRCDUsersInContact() {
    $db = $this->_db;
    $userId = $this->getId();
    $status = RCD_in_contact;

    $q = "
      (
      SELECT a.user_id AS to_user_id, 'CONTACT' AS
      STATUS FROM users a
      JOIN rcd b ON a.user_id = b.user_id
      WHERE b.to_user_id =? AND a.user_id <> ? AND b.status_id = $status
      )
      UNION (

      SELECT a.user_id AS to_user_id, 'REF_CONTACT' AS
      STATUS FROM users a
      JOIN rcd b ON a.user_id = b.to_user_id
      WHERE b.user_id = ? AND a.user_id <> ? AND b.status_id = $status
      )
   ";

   return $db->fetchAssoc($q, array($userId, $userId, $userId, $userId));
  }

  public function getRCDUsersSent($status) {
    $db = $this->_db;
    $select = $db->select();

    $select->from('rcd', 'to_user_id');
    $select->where('user_id = ?', $this->getId());
    $select->where('status_id = ?', RCD_sent);

    return $db->fetchAssoc($select);
  }

  public function getRCDUsersReceived($status) {
    $db = $this->_db;
    $select = $db->select();

    $select->from('rcd', 'user_id as to_user_id');
    $select->where('to_user_id = ?', $this->getId());
    $select->where('status_id = ?', RCD_sent);

    return $db->fetchAssoc($select);
  }

  public function acceptUser($user2AcceptId) {
    $db = $this->_db;
    $userId = $this->getId();

    $set = array('status_id' => RCD_in_contact);
    $where = $db->quoteInto("status_id=?", RCD_sent);
    $where.= $db->quoteInto("AND user_id=?", $user2AcceptId);
    $where.= $db->quoteInto("AND to_user_id=?", $userId);

    $res = $db->update('rcd', $set, $where);
  }

  public function rejectUser($user2RejectId) {
    $db = $this->_db;
    $userId = $this->getId();

    $where = $db->quoteInto("status_id=?", RCD_sent);
    $where.= $db->quoteInto("AND user_id=?", $user2RejectId);
    $where.= $db->quoteInto("AND to_user_id=?", $userId);
    
    // update the table and get the number of rows affected
    $res = $db->delete('rcd', $where);    
  }

  public function sendRequestTo($toUserId) {
    $db = $this->getDb();
    $userId = $this->getId();

    $this->setRelation($toUserId, 'to_user_id', 'rcd', array('status_id' => RCD_sent));

    return true;
  }

  /**
   * Checks whether the user associated with the given $userId is in contact with this user.
   *
   * @param int $userId
   * @return boolean
   */
  public function isInContact($userId) {
    if($userId == $this->getId()) return true;
    $rowArr = $this->getRCDUsersInContact();
    if(!is_array($rowArr)) return false;
    foreach($rowArr as $k => $currRow) {
      if($currRow['to_user_id'] == $userId) return true;
    }
    return false;
  }

  public static function authenticate($username, $password) {
    // check if the user is registered via looking up in db
    $userMod = UserModel::getInstanceByUsername($username);
    if($userMod == null) return false;
    // now check if the password matches...
    if($userMod->password != $password) return false;

    $_SESSION['user_id'] = $userMod->getId();
    $_SESSION['last_login'] = $userMod->last_login;

    $userMod->last_login = time();
    $userMod->save();
    return true;
  }
  
  /**
   * creates a new user from a data container
   *
   * @param D100_Container $container
   * @return UserModel
   */
  public static function create(D100_Container $container) {
    // check if the user's username is already registered via looking up in db
    $userMod = UserModel::getInstanceByUsername($container->user_name);
    if($userMod != null) return -1;

    require_once 'GEOCode.php';
    
    try {
      if($container->geo_code != "") {
        $convertedCode = GEOCode::convertStr2GEOCode($container->geo_code);
      }
      else $convertedCode = array(null, null);
    }
    catch (Exception $e) {
      return -2;
    }
    
    // create new user and store it in db

    $userMod = UserModel::getInstance();

    $userMod->geo_lat = $convertedCode[0];
    $userMod->geo_long = $convertedCode[1];
    
    $userMod->password = $container->password;
    $userMod->user_name = $container->user_name;
    $userMod->full_name = $container->full_name;
    $userMod->email = $container->email;
    $userMod->town = $container->town;
    $userMod->country = $container->country;
    $userMod->life_motto = $container->life_motto;
    $userMod->secondary_life_motto = $container->secondary_life_motto;

    $userMod->creation_date = time();

    $userMod->save();
    return $userMod;
  }
}
?>