/*******************************************************************************
 * Copyright (c) 2002-2006 Innoopract Informationssysteme GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Innoopract Informationssysteme GmbH - initial API and implementation
 ******************************************************************************/

package org.platforms.contest.core.model.internal.db;

import java.util.*;
import org.platforms.contest.core.model.IUser;
import org.platforms.contest.core.model.SearchCriterias;
import org.platforms.contest.core.model.internal.*;


public final class FakeDB {
  
  // key: id, value: User
  private final Map users = new HashMap(); 
  // key: id, value: UserRelation
  private final Map userRelations = new HashMap();
  // key: id, value: TTT
  private final Map ttts = new HashMap();  
  
  ///////////////////
  // Mutating methods
  
  public void save( final IUser user ) {
    User userImpl = ( User )user;
    ModelObject userEntity = copy( userImpl );
    users.put( userEntity.getId(), userEntity );
    TTT ttt = ( TTT )user.getTTT();
    if( ttt != null ) {
      TTT tttEntity = copy( ttt );
      ttts.put( tttEntity.getId(), tttEntity );
    }
  }

  public void sendRCD( final IUser sendingUser, final IUser receivingUser ) {
    IUser[] users = findReceived( sendingUser );
    boolean hasReceived = false;
    for( int i = 0; !hasReceived && i < users.length; i++ ) {
      if( getId( users[ i ] ).equals( getId( receivingUser ) ) ) {
        hasReceived = true;
      }
    }
    if( hasReceived ) {
      confirmRCD( sendingUser, receivingUser );
    } else {
      UserRelation userRelation = new UserRelation();
      User userImp = ( User )sendingUser;
      userRelation.setSenderId( userImp.getId() );
      User toUserImpl = ( User )receivingUser;
      userRelation.setReceiverId( toUserImpl.getId() );
      userRelation.setContactState( ContactState.RCD_SENT );
      userRelations.put( userRelation.getId(), userRelation );
    }
  }
  
  public void confirmRCD( final IUser receivingUser, final IUser sendingUser ) {
    Iterator iter = userRelations.values().iterator();
    UserRelation userRelation = null;
    while( userRelation == null && iter.hasNext() ) {
      UserRelation currentUserRelation = ( UserRelation )iter.next();
      String sendingUserId = getId( sendingUser );
      String receivingUserId = getId( receivingUser );
      if(    currentUserRelation.getSenderId().equals( sendingUserId ) 
          && currentUserRelation.getReceiverId().equals( receivingUserId ) )
      {
        userRelation = currentUserRelation;
      }
    }
    if(    userRelation != null 
        && ContactState.RCD_SENT.equals( userRelation.getContactState() )) 
    {
      userRelation.setContactState( ContactState.IN_CONTACT );
    }
  }
  
  ///////////////
  // Find methods
  
  public IUser findByLoginName( final String loginName ) {
    Iterator iter = users.values().iterator();
    User result = null;
    while( result == null && iter.hasNext() ) {
      User user = ( User )iter.next();
      if( user.getLoginName().equals( loginName ) ) {
        result = copy( user );
        resolveTTT( result );
      }
    }
    return result;
  }
  
  public IUser[] findSent( final IUser user ) {
    User userImpl = ( User )user;
    String userId = userImpl.getId();
    List inContact = new ArrayList();
    Iterator iter = userRelations.values().iterator();
    while( iter.hasNext() ) {
      UserRelation userRelation = ( UserRelation )iter.next();
      if(    userRelation.getContactState().equals( ContactState.RCD_SENT ) 
          && userRelation.getSenderId().equals( userId ) ) 
      {
        inContact.add( findUserById( userRelation.getContactFor( userImpl ) ) );
      }
    }
    IUser[] result = new IUser[ inContact.size() ];
    inContact.toArray( result );
    return result;
  }
  
  public IUser[] findReceived( final IUser user ) {
    User userImpl = ( User )user;
    String userId = userImpl.getId();
    List inContact = new ArrayList();
    Iterator iter = userRelations.values().iterator();
    while( iter.hasNext() ) {
      UserRelation userRelation = ( UserRelation )iter.next();
      if(    userRelation.getContactState().equals( ContactState.RCD_SENT ) 
          && userRelation.getReceiverId().equals( userId ) ) 
      {
        inContact.add( findUserById( userRelation.getContactFor( userImpl ) ) );
      }
    }
    IUser[] result = new IUser[ inContact.size() ];
    inContact.toArray( result );
    return result;
  }

  public IUser[] findInContact( IUser user ) {
    List inContact = new ArrayList();
    Iterator iter = userRelations.values().iterator();
    while( iter.hasNext() ) {
      UserRelation userRelation = ( UserRelation )iter.next();
      if(    userRelation.getContactState().equals( ContactState.IN_CONTACT ) 
          && userRelation.getContactFor( user ) != null ) 
      {
        inContact.add( findUserById( userRelation.getContactFor( user ) ) );
      }
    }
    IUser[] result = new IUser[ inContact.size() ];
    inContact.toArray( result );
    return result;
  }

  public IUser[] search( final IUser user, final SearchCriterias criterias ) {
    List list = new ArrayList();
    Iterator iter = users.values().iterator();
    while( iter.hasNext() ) {
      IUser otherUser = ( IUser )iter.next();
      if(    !getId( user ).equals( getId( otherUser ) ) 
          && !isInContact( user, otherUser ) ) 
      {
        list.add( otherUser );
      }
    }
    IUser[] result = new IUser[ list.size() ];
    list.toArray( result );
    return result;
  }

  public boolean isInContact( final IUser user1, final IUser user2 ) {
    boolean result = false;
    String user1Id = getId( user1 );
    String user2Id = getId( user2 );
    Iterator iter = userRelations.values().iterator();
    while( !result && iter.hasNext() ) {
      UserRelation userRelation = ( UserRelation )iter.next();
      String receiverId = userRelation.getReceiverId();
      String senderId = userRelation.getSenderId();
      if(    userRelation.getContactState().equals( ContactState.IN_CONTACT ) 
          && (    ( senderId.equals( user2Id ) && receiverId.equals( user1Id ) )
               || ( senderId.equals( user1Id ) && receiverId.equals( user2Id ) ) ) )
      {
        result = true;
      }
    }
    return result;
  }

  public void close() {
    // do nothing
  }

  private IUser findUserById( final String id ) {
    return ( IUser )users.get( id );
  }
  
  private void resolveTTT( final User user ) {
    String tttId = user.getTTTId();
    TTT ttt = ( TTT )ttts.get( tttId );
    user.setTTT( ttt );
  }
  
  private User copy( final User user ) {
    User result = new User();
    result.setId( user.getId() );
    result.setLoginName( user.getLoginName() );
    result.setPassword( user.getPassword() );
    result.setEMail( user.getEMail() );
    result.setCountry( user.getCountry() );
    result.setFullname( user.getFullname() );
    result.setMotto( user.getMotto() );
    result.setTown( user.getTown() );
    result.setTTT( copy( ( TTT )user.getTTT() ) );
    return result;
  }
  
  private TTT copy ( final TTT ttt ) {
    TTT result = null;
    if( ttt != null ) {
      result = new TTT();
      result.setId( ttt.getId() );
    }
    return result;
  }
  
  private static String getId( final IUser user ) {
    User userImpl = ( User )user;
    return userImpl.getId();
  }
}
